require('dotenv').config();
const express = require('express');
const http = require('http');
const { Server } = require('socket.io');
const cloudinary = require('cloudinary').v2;
const cors = require('cors');
const axios = require('axios');
const multer = require('multer');
const path = require('path');
const fs = require('fs');

const app = express();
app.use(cors());
app.use(express.json());

process.on('uncaughtException', (err) => {
  console.error('UNCAUGHT EXCEPTION! 💥 Shutting down...');
  console.error(err.name, err.message);
  console.error(err.stack);
  process.exit(1);
});

// Configure temporary storage for uploads
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const uploadDir = path.join(__dirname, 'uploads');
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir);
    }
    cb(null, uploadDir);
  },
  filename: function (req, file, cb) {
    cb(null, Date.now() + '-' + file.originalname);
  }
});

const upload = multer({ storage: storage });

// Configure Cloudinary
cloudinary.config({
  cloud_name: process.env.CLOUDINARY_CLOUD_NAME || 'your_cloud_name',
  api_key: process.env.CLOUDINARY_API_KEY || 'your_api_key',
  api_secret: process.env.CLOUDINARY_API_SECRET || 'your_api_secret',
});

const server = http.createServer(app);

const io = new Server(server, {
  cors: {
    origin: process.env.FRONTEND_URL,
    methods: ['GET', 'POST'],
  },
});

// File upload endpoint
app.post('/upload', upload.single('file'), async (req, res) => {
  try {
    // Extract token from headers
    const authHeader = req.headers.authorization;
    if (!authHeader || !authHeader.startsWith('Bearer ')) {
      return res.status(401).json({ error: 'Unauthorized: No token provided' });
    }
    
    const token = authHeader.substring(7); // Remove 'Bearer ' prefix
    
    // Validate user with Laravel backend
    try {
      const response = await axios.get(`${process.env.LARAVEL_API_URL}/validate-token`, {
        headers: {
          Authorization: `Bearer ${token}`,
        },
      });
      
      if (!response.data.user) {
        return res.status(401).json({ error: 'Unauthorized: Invalid token' });
      }
    } catch (error) {
      console.error('Token validation error:', error.message);
      return res.status(401).json({ error: 'Unauthorized: Failed to validate token' });
    }
    
    // File exists?
    if (!req.file) {
      return res.status(400).json({ error: 'No file uploaded' });
    }
    
    // Upload file to Cloudinary
    const result = await cloudinary.uploader.upload(req.file.path, {
      resource_type: 'auto',
    });
    
    // Delete the temporary file
    fs.unlinkSync(req.file.path);
    
    // Return the file URL
    return res.status(200).json({
      fileUrl: result.secure_url,
      fileName: req.file.originalname,
    });
  } catch (error) {
    console.error('File upload error:', error);
    return res.status(500).json({ error: 'File upload failed' });
  }
});

// Middleware to validate user token
io.use(async (socket, next) => {
  const token = socket.handshake.auth.token;
  //console.log('Token received:', token);

  if (!token) {
    console.error('Authentication error: No token provided');
    return next(new Error('Authentication error: No token provided'));
  }

  try {
    // Validate token with Laravel backend
    const response = await axios.get(`${process.env.LARAVEL_API_URL}/validate-token`, {
      headers: {
        Authorization: `Bearer ${token}`,
      },
    });

    console.log('Validation response:', response.data);

    if (response.data.user) {
      socket.user = response.data.user; // Attach user data to the socket
      console.log(`User authenticated: ${socket.user.name}`);
      // After successful authentication:
      if (socket.user && socket.user.id) {
        userSocketMap.set(socket.user.id, socket.id);
      }
      next();
    } else {
      console.error('Authentication error: Invalid token');
      return next(new Error('Authentication error: Invalid token'));
    }
  } catch (error) {
    console.error('Validation error:', error.message);
    console.error('Error details:', error.response ? error.response.data : 'No response data');
    return next(new Error('Authentication error: Failed to validate token'));
  }
});

// Store active rooms and their members
const activeRooms = new Map();

// Simple rate limiter for the typing events
const typingThrottleMap = new Map();
const userStatus = new Map(); // Maps user IDs to their status
const userSocketMap = new Map(); // Maps user IDs to their socket IDs

// Socket.IO connection
io.on('connection', (socket) => {
  console.log('A user connected:', socket.user.name);

  

  // Join a room
  socket.on('joinRoom', (roomId) => {
    console.log(`User ${socket.user.name} attempting to join room ${roomId}`);
    socket.join(roomId);
    console.log(`User ${socket.user.name} joined room ${roomId}`);

    // Add user to the active rooms map
    if (!activeRooms.has(roomId)) {
      activeRooms.set(roomId, new Set());
    }
    activeRooms.get(roomId).add(socket.user.orbit_id);

    // Notify room members about the new user
    io.to(roomId).emit('userJoined', {
      user: socket.user,
      roomId,
    });
  });

  // Set user as online by default when connecting
  if (socket.user && socket.user.id) {
    userStatus.set(socket.user.id, 'online');
    
    // Broadcast to everyone that this user is online
    io.emit('usersStatusUpdate', { [socket.user.id]: 'online' });
  }

  // Handle status update requests
  socket.on('requestAllUsersStatus', () => {
      // Convert Map to a plain object for sending via socket
      const statusObject = {};
      userStatus.forEach((status, userId) => {
          statusObject[userId] = status;
      });
      
      console.log('Sending all user statuses:', statusObject);
      
      // Send current status to the requesting client
      socket.emit('usersStatusUpdate', statusObject);
  });

  // Handle status updates
  socket.on('updateStatus', ({ status }) => {
      console.log(`User ${socket.user.name} updated status to: ${status}`);
      
      // Update status in our map
      if (socket.user && socket.user.id) {
          userStatus.set(socket.user.id, status);
      }
      
      // Broadcast to ALL connected users (including sender)
      io.emit('usersStatusUpdate', { [socket.user.id]: status });
  });



  // Delete Message
  socket.on('deleteMessage', async ({ messageId, roomId }) => {
    try {
      const response = await axios.delete(
        `${process.env.LARAVEL_API_URL}/messages/${messageId}`,
        {
          headers: {
            Authorization: `Bearer ${socket.handshake.auth.token}`,
          },
        }
      );
      
      if (response.data.success) {
        // Broadcast to room that message has been deleted
        io.to(roomId).emit('messageDeleted', { messageId });
      }
    } catch (error) {
      console.error('Error deleting message:', error.message);
    }
  });

  // Add this to the socket.io connection block in server.js
  socket.on('deleteFile', async ({ messageId, fileUrl, roomId }) => {
    try {
      // First delete from Cloudinary if it's a Cloudinary URL
      if (fileUrl && fileUrl.includes('cloudinary.com')) {
        // Extract public_id from the Cloudinary URL
        const urlParts = fileUrl.split('/');
        const filenameWithExtension = urlParts[urlParts.length - 1];
        const publicId = filenameWithExtension.split('.')[0];
        
        // Delete from Cloudinary
        await cloudinary.uploader.destroy(publicId);
      }
      
      // Then delete from database via Laravel API
      const response = await axios.delete(
        `${process.env.LARAVEL_API_URL}/messages/${messageId}`,
        {
          headers: {
            Authorization: `Bearer ${socket.handshake.auth.token}`,
          },
        }
      );
      
      if (response.data.success) {
        // Broadcast to room that file has been deleted
        io.to(roomId).emit('messageDeleted', { messageId });
      }
    } catch (error) {
      console.error('Error deleting file:', error.message);
    }
  });

  // Edit message
  socket.on('editMessage', async ({ messageId, newText, roomId }) => {
    try {
      const response = await axios.put(
        `${process.env.LARAVEL_API_URL}/messages/${messageId}`,
        {
          message: newText,
        },
        {
          headers: {
            Authorization: `Bearer ${socket.handshake.auth.token}`,
          },
        }
      );
      
      if (response.data.message) {
        io.to(roomId).emit('messageEdited', response.data.message);
      }
    } catch (error) {
      console.error('Error editing message:', error.message);
    }
  });


  // Leave a room
  socket.on('leaveRoom', (roomId) => {
    console.log(`User ${socket.user.name} attempting to leave room ${roomId}`);
    socket.leave(roomId);
    console.log(`User ${socket.user.name} left room ${roomId}`);

    // Remove user from the active rooms map
    if (activeRooms.has(roomId)) {
      activeRooms.get(roomId).delete(socket.user.orbit_id);
      if (activeRooms.get(roomId).size === 0) {
        activeRooms.delete(roomId); // Delete room if empty
      }
    }

    // Notify room members about the user leaving
    io.to(roomId).emit('userLeft', {
      user: socket.user,
      roomId,
    });
  });


// Send message to a room
socket.on('sendMessage', async ({ roomId, message, replyToId }, callback) => {
  console.log(`Message received from ${socket.user.name} in room ${roomId}: ${message}`);

  try {
    // Extract user IDs from the room ID (assuming format like "1-2")
    let senderId = socket.user.id;
    let receiverId;
    
    if (roomId.includes('-')) {
      // If roomId is in the format "1-2", parse it to get the other user's ID
      const userIds = roomId.split('-').map(id => parseInt(id, 10));
      receiverId = userIds[0] === senderId ? userIds[1] : userIds[0];
    } else {
      // Fallback to treating roomId as receiverId directly
      receiverId = parseInt(roomId, 10);
    }

    console.log('Attempting to save message to Laravel backend...');
    const response = await axios.post(
      `${process.env.LARAVEL_API_URL}/save-message`,
      {
        sender_id: senderId,
        receiver_id: receiverId,
        message: message,
        reply_to_id: replyToId ? replyToId : null, // Include reply reference
      },
      {
        headers: {
          Authorization: `Bearer ${socket.handshake.auth.token}`,
        },
      }
    );

    console.log('Message saved successfully:', response.data);

    const newMessage = response.data.message;

    // Emit the message to the room with consistent data structure
    console.log(`Emitting message to room ${roomId}`);
    io.to(roomId).emit('receiveMessage', {
      id: newMessage.id,
      sender_id: socket.user.id,
      user: socket.user,
      message: newMessage.message,
      reply_to_id: newMessage.reply_to_id, // Include reply ID
      type: 'text',
      created_at: newMessage.created_at,
    });

    // Send acknowledgment to sender
    if (callback) {
      callback({ success: true, messageId: newMessage.id });
    }

    // Create notification for receiver
    await axios.post(
      `${process.env.LARAVEL_API_URL}/create-notification`,
      {
        user_id: receiverId,
        sender_id: senderId,
        message_id: newMessage.id,
        type: 'private_message',
        data: {
          message: message,
          sender: socket.user.name
        }
      },
      {
        headers: {
          Authorization: `Bearer ${socket.handshake.auth.token}`,
        },
      }
    );

    // Check if receiver is online
    const receiverSocket = findUserSocket(receiverId);
    if (!receiverSocket) {
      // Send push notification if user is offline
      await axios.post(
        `${process.env.LARAVEL_API_URL}/send-push-notification`,
        {
          user_id: receiverId,
          title: 'New message',
          body: `${socket.user.name}: ${message}`,
          data: {
            type: 'private_message',
            sender_id: senderId,
            message_id: newMessage.id
          }
        },
        {
          headers: {
            Authorization: `Bearer ${socket.handshake.auth.token}`,
          },
        }
      );
    }

    io.to(`user:${receiverId}`).emit('updateUnreadCounts');
    


  } catch (error) {
    console.error('Error saving message to database:', error.message);
    console.error('Error details:', error.response ? error.response.data : 'No response data');
    
    // Send error back to client
    if (callback) {
      callback({ error: 'Failed to save message' });
    }
  }
});

// Update the sendFile handler to include reply support
socket.on('sendFile', async ({ roomId, fileUrl, fileName, replyToId }) => {
  console.log(`File received from ${socket.user.name} in room ${roomId}: ${fileName}`);
  
  try {
    // Extract user IDs from room ID
    let senderId = socket.user.id;
    let receiverId;
    
    if (roomId.includes('-')) {
      // If roomId is in the format "1-2", parse it
      const userIds = roomId.split('-').map(id => parseInt(id, 10));
      receiverId = userIds[0] === senderId ? userIds[1] : userIds[0];
    } else {
      receiverId = parseInt(roomId, 10);
    }
    
    // Save file message to database
    console.log('Attempting to save file message to Laravel backend...');
    const response = await axios.post(
      `${process.env.LARAVEL_API_URL}/save-file-message`,
      {
        sender_id: senderId,
        receiver_id: receiverId,
        message: `File: ${fileName}`,
        file_url: fileUrl,
        file_name: fileName,
        reply_to_id: replyToId ? replyToId : null, // Include reply ID for files
      },
      {
        headers: {
          Authorization: `Bearer ${socket.handshake.auth.token}`,
        },
      }
    );

    console.log('File message saved successfully:', response.data);
    const newFileMessage = response.data.message;

    // Emit the file to the room
    io.to(roomId).emit('receiveFile', {
      id: newFileMessage.id,
      user: socket.user,
      sender_id: socket.user.id,
      message: `File: ${fileName}`,
      type: 'file',
      fileUrl: fileUrl,
      fileName: fileName,
      reply_to_id: newFileMessage.reply_to_id, // Include reply ID
      created_at: newFileMessage.created_at
    });

    // Send a notification about the file
    io.to(roomId).emit('newMessageNotification', {
      sender: socket.user,
      message: `Sent a file: ${fileName}`,
    });
  } catch (error) {
    console.error('Error saving file message to database:', error.message);
    console.error('Error details:', error.response ? error.response.data : 'No response data');
  }
});

  // Handle typing indicator
  socket.on('typing', ({ roomId, isTyping }) => {
    const userId = socket.user.id;
    const key = `${userId}-${roomId}`;
    const now = Date.now();
    
    // Only process if it's been at least 2 seconds since last event
    // or if typing status changed
    if (!typingThrottleMap.has(key) || 
        now - typingThrottleMap.get(key).time > 2000 ||
        typingThrottleMap.get(key).status !== isTyping) {
        
        console.log(`User ${socket.user.name} is ${isTyping ? 'typing' : 'not typing'} in room ${roomId}`);
        
        // Update throttle map
        typingThrottleMap.set(key, { time: now, status: isTyping });
        
        // Broadcast to everyone in the room EXCEPT the sender
        socket.to(roomId).emit('userTyping', {
            user: socket.user,
            isTyping,
            timestamp: now
        });
    }
  });

  // Handle message read receipts
  socket.on('markAsRead', async ({ messageIds, roomId }) => {
    if (!messageIds || messageIds.length === 0) {
        console.log('No message IDs provided to markAsRead');
        return;
    }
    
    console.log(`User ${socket.user.name} marked messages as read:`, messageIds);
    
    try {
        const response = await axios.post(
            `${process.env.LARAVEL_API_URL}/mark-messages-read`,
            {
                message_ids: messageIds,
                user_id: socket.user.id
            },
            {
                headers: {
                    Authorization: `Bearer ${socket.handshake.auth.token}`,
                },
            }
        );
        
        if (response.data.success) {
            console.log('Messages marked as read:', response.data);
            
            // Notify all users in the room that messages were read
            io.to(roomId).emit('messagesRead', {
                reader: socket.user.id,
                messageIds
            });
        }
    } catch (error) {
        console.error('Error marking messages as read:', error.message);
    }
  });
  

  // START OF GROUP CHAT IMPLEMENTATION
  socket.on('createGroup', async (groupData) => {
    try {
      const response = await axios.post(
        `${process.env.LARAVEL_API_URL}/groups`,
        groupData,
        {
          headers: {
            Authorization: `Bearer ${socket.handshake.auth.token}`,
          },
        }
      );
      
      socket.emit('groupCreated', response.data.group);
    } catch (error) {
      console.error('Error creating group:', error.message);
      socket.emit('error', { message: 'Failed to create group' });
    }
  });

  socket.on('joinGroup', async (groupId) => {
    console.log(`User ${socket.user.name} joining group ${groupId}`);
    
    // Check if the socket is already in the room to prevent duplicate joins
    const rooms = Object.keys(socket.rooms);
    const roomName = `group:${groupId}`;
    
    if (!rooms.includes(roomName)) {
      socket.join(roomName);
      
      // Notify group members only if this is a new join
      socket.to(roomName).emit('userJoinedGroup', {
        user: socket.user,
        groupId
      });
    }
  });

  socket.on('leaveGroup', (groupId) => {
    console.log(`User ${socket.user.name} leaving group ${groupId}`);
    socket.leave(`group:${groupId}`);
    
    // Notify group members
    socket.to(`group:${groupId}`).emit('userLeftGroup', {
      user: socket.user,
      groupId
    });
  });

  socket.on('sendGroupMessage', async ({ groupId, message, replyToId }) => {
    try {
      const response = await axios.post(
        `${process.env.LARAVEL_API_URL}/group-messages`,
        {
          group_id: groupId,
          message: message,
          reply_to_id: replyToId
        },
        {
          headers: {
            Authorization: `Bearer ${socket.handshake.auth.token}`,
          },
        }
      );
      
      const newMessage = response.data.message;
      
      // Broadcast to group members
      io.to(`group:${groupId}`).emit('receiveGroupMessage', {
        id: newMessage.id,
        user_id: socket.user.id,
        user: socket.user,
        message: newMessage.message,
        reply_to_id: newMessage.reply_to_id,
        group_id: groupId,
        created_at: newMessage.created_at
      });

      // Get all group members except sender
      const res = await axios.get(
        `${process.env.LARAVEL_API_URL}/group-members/${groupId}`,
        {
          headers: {
            Authorization: `Bearer ${socket.handshake.auth.token}`,
          },
        }
      );
      
      const members = res.data.members.filter(m => m.id !== socket.user.id);
      
      // Create notifications for each member
      await Promise.all(members.map(async member => {
        await axios.post(
          `${process.env.LARAVEL_API_URL}/create-notification`,
          {
            user_id: member.id,
            sender_id: socket.user.id,
            group_id: groupId,
            group_message_id: newMessage.id,
            type: 'group_message',
            data: {
              message: message,
              sender: socket.user.name,
              group: groupId
            }
          },
          {
            headers: {
              Authorization: `Bearer ${socket.handshake.auth.token}`,
            },
          }
        );

        // Check if member is online
        const memberSocket = findUserSocket(member.id);
        if (!memberSocket) {
          // Send push notification if user is offline
          await axios.post(
            `${process.env.LARAVEL_API_URL}/send-push-notification`,
            {
              user_id: member.id,
              title: `New message in ${groupName}`,
              body: `${socket.user.name}: ${message}`,
              data: {
                type: 'group_message',
                group_id: groupId,
                message_id: newMessage.id
              }
            },
            {
              headers: {
                Authorization: `Bearer ${socket.handshake.auth.token}`,
              },
            }
          );
        }
      }));

      members.forEach(member => {
        io.to(`user:${member.id}`).emit('updateUnreadCounts');
      });

    } catch (error) {
      console.error('Error sending group message:', error.message);
      socket.emit('error', { message: 'Failed to send message' });
    }
  });

  socket.on('sendGroupFile', async ({ groupId, fileUrl, fileName, replyToId }) => {
    try {
      const response = await axios.post(
        `${process.env.LARAVEL_API_URL}/group-file-messages`,
        {
          group_id: groupId,
          message: `File: ${fileName}`,
          file_url: fileUrl,
          file_name: fileName,
          reply_to_id: replyToId
        },
        {
          headers: {
            Authorization: `Bearer ${socket.handshake.auth.token}`,
          },
        }
      );
      
      const newMessage = response.data.message;
      
      // Broadcast to group members
      io.to(`group:${groupId}`).emit('receiveGroupFile', {
        id: newMessage.id,
        user_id: socket.user.id,
        user: socket.user,
        message: newMessage.message,
        file_url: fileUrl,
        file_name: fileName,
        reply_to_id: newMessage.reply_to_id,
        group_id: groupId,
        type: 'file',
        created_at: newMessage.created_at
      });
    } catch (error) {
      console.error('Error sending group file:', error.message);
      socket.emit('error', { message: 'Failed to send file' });
    }
  });

  socket.on('markGroupMessagesAsRead', async ({ messageIds, groupId }) => {
    try {
      await axios.post(
        `${process.env.LARAVEL_API_URL}/group-messages/mark-as-read`,
        {
          message_ids: messageIds,
          group_id: groupId
        },
        {
          headers: {
            Authorization: `Bearer ${socket.handshake.auth.token}`,
          },
        }
      );
      
      // Inform group members that messages have been read
      io.to(`group:${groupId}`).emit('groupMessagesRead', {
        messageIds,
        reader: socket.user.id,
        groupId
      });
    } catch (error) {
      console.error('Error marking group messages as read:', error.message);
    }
  });
  

  socket.on('groupTyping', ({ groupId, isTyping }) => {
      const userId = socket.user.id;
      const key = `${userId}-group-${groupId}`;
      const now = Date.now();

      if (!typingThrottleMap.has(key) || 
          now - typingThrottleMap.get(key).time > 2000 ||
          typingThrottleMap.get(key).status !== isTyping) {
          
            console.log(`User ${socket.user.name} is ${isTyping ? 'typing' : 'not typing'} in group ${groupId}`);

          // Update throttle map
          typingThrottleMap.set(key, { time: now, status: isTyping });
          
          // Broadcast to everyone in the group EXCEPT the sender
          socket.to(groupId).emit('userGroupTyping', {
              user: socket.user,
              isTyping,
              groupId,
              timestamp: now
          });
      }
  });

  socket.on('fileDeleted', (data) => {
    // Broadcast to everyone in the group that a file was deleted
    socket.to(data.groupId).emit('groupFileDeleted', {
      messageId: data.messageId,
      groupId: data.groupId,
      deletedBy: socket.user.id
    });
  });

  // Handle message deletion
  socket.on('deleteGroupMessage', async ({ messageId, groupId }) => {
    try {
      // Notify all users in the group about the deleted message
      socket.to(groupId).emit('groupMessageDeleted', { messageId, groupId });
    } catch (error) {
      console.error('Error handling message deletion:', error);
    }
  });

  // Handle message editing
  socket.on('updateGroupMessage', async ({ messageId, groupId, newMessage }) => {
    try {
      // Notify all users in the group about the updated message
      socket.to(groupId).emit('groupMessageUpdated', { 
        messageId, 
        groupId, 
        message: newMessage,
        edited: true
      });
    } catch (error) {
      console.error('Error handling message update:', error);
    }
  });

  // Listen for new forwarded messages
  socket.on('sendForwardedMessage', async (data) => {
    try {
      const { recipientId, message, forwardedFrom, tempId } = data;
      
      // Broadcast to the recipient
      io.to(`user:${recipientId}`).emit('receivedMessage', {
        id: tempId,
        user_id: socket.user.id,
        message,
        forwarded_from: forwardedFrom,
        created_at: new Date().toISOString()
      });
      
      // Acknowledge the sender
      io.to(`user:${socket.user.id}`).emit('messageSent', {
        tempId,
        status: 'success'
      });
    } catch (error) {
      console.error('Error handling forwarded message:', error);
    }
  });

  // Listen for new forwarded files
  socket.on('sendForwardedFile', async (data) => {
    try {
      const { recipientId, fileUrl, fileName, forwardedFrom, tempId } = data;
      
      // Broadcast to the recipient
      io.to(`user:${recipientId}`).emit('receivedFile', {
        id: tempId,
        user_id: socket.user.id,
        file_url: fileUrl,
        file_name: fileName,
        forwarded_from: forwardedFrom,
        type: 'file',
        created_at: new Date().toISOString()
      });
      
      // Acknowledge the sender
      io.to(`user:${socket.user.id}`).emit('fileSent', {
        tempId,
        status: 'success'
      });
    } catch (error) {
      console.error('Error handling forwarded file:', error);
    }
  });

  // Listen for new forwarded group messages
  socket.on('sendForwardedGroupMessage', async (data) => {
    try {
      const { groupId, message, forwardedFrom, tempId } = data;
      
      // Broadcast to the group
      io.to(`group:${groupId}`).emit('receivedGroupMessage', {
        id: tempId,
        user_id: socket.user.id,
        message,
        group_id: groupId,
        forwarded_from: forwardedFrom,
        created_at: new Date().toISOString()
      });
      
      // Acknowledge the sender
      io.to(`user:${socket.user.id}`).emit('groupMessageSent', {
        tempId,
        status: 'success'
      });
    } catch (error) {
      console.error('Error handling forwarded group message:', error);
    }
  });

  // Listen for new forwarded group files
  socket.on('sendForwardedGroupFile', async (data) => {
    try {
      const { groupId, fileUrl, fileName, forwardedFrom, tempId } = data;
      
      // Broadcast to the group
      io.to(`group:${groupId}`).emit('receivedGroupFile', {
        id: tempId,
        user_id: socket.user.id,
        file_url: fileUrl,
        file_name: fileName,
        group_id: groupId,
        forwarded_from: forwardedFrom,
        type: 'file',
        created_at: new Date().toISOString()
      });
      
      // Acknowledge the sender
      io.to(`user:${socket.user.id}`).emit('groupFileSent', {
        tempId,
        status: 'success'
      });
    } catch (error) {
      console.error('Error handling forwarded group file:', error);
    }
  });

  // END OF GROUP IMPLEMENTATION

  // Disconnect event
  socket.on('disconnect', () => {
    console.log(`User disconnected: ${socket.user ? socket.user.name : 'Unknown user'}`);

    
    
    // Remove user from all active rooms
    activeRooms.forEach((users, roomId) => {
      if (socket.user && users.has(socket.user.orbit_id)) {
        users.delete(socket.user.orbit_id);
        if (users.size === 0) {
          activeRooms.delete(roomId);
        }
        
        // Notify room about user disconnect
        io.to(roomId).emit('userLeft', {
          user: socket.user,
          roomId,
          reason: 'disconnected'
        });
      }
    });

    // Set user as offline
    if (socket.user && socket.user.id) {
      userStatus.set(socket.user.id, 'offline');
      
      // Broadcast offline status to all users
      const statusUpdate = { [socket.user.id]: 'offline' };
      io.emit('usersStatusUpdate', statusUpdate);
    }
  });
});

// Health check endpoint
app.get('/health', (req, res) => {
  res.status(200).json({ status: 'healthy', uptime: process.uptime() });
});

// Get active users in a room
app.get('/api/room/:roomId/users', async (req, res) => {
  try {
    const roomId = req.params.roomId;
    
    // Check if room exists
    if (!activeRooms.has(roomId)) {
      return res.status(404).json({ error: 'Room not found' });
    }
    
    // Get user IDs from the room
    const userIds = Array.from(activeRooms.get(roomId));
    
    // Return active users
    res.status(200).json({ users: userIds });
  } catch (error) {
    console.error('Error retrieving room users:', error);
    res.status(500).json({ error: 'Failed to retrieve room users' });
  }
});

// Set server port
const PORT = process.env.PORT || 5000;

// Start server
server.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
}).on('error', (err) => {
  console.error('Server error:', err);
});